/*
 * Decompiled with CFR 0.152.
 */
package org.apache.curator.framework.recipes.cache;

import com.google.common.base.Function;
import java.io.Closeable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.WatcherRemoveCuratorFramework;
import org.apache.curator.framework.api.BackgroundCallback;
import org.apache.curator.framework.api.BackgroundPathable;
import org.apache.curator.framework.api.CuratorEvent;
import org.apache.curator.framework.api.ErrorListenerPathable;
import org.apache.curator.framework.api.GetDataWatchBackgroundStatable;
import org.apache.curator.framework.api.UnhandledErrorListener;
import org.apache.curator.framework.listen.Listenable;
import org.apache.curator.framework.listen.ListenerContainer;
import org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.curator.framework.recipes.cache.DefaultTreeCacheSelector;
import org.apache.curator.framework.recipes.cache.TreeCacheEvent;
import org.apache.curator.framework.recipes.cache.TreeCacheListener;
import org.apache.curator.framework.recipes.cache.TreeCacheSelector;
import org.apache.curator.framework.state.ConnectionState;
import org.apache.curator.framework.state.ConnectionStateListener;
import org.apache.curator.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.curator.shaded.com.google.common.base.Preconditions;
import org.apache.curator.shaded.com.google.common.collect.ImmutableMap;
import org.apache.curator.shaded.com.google.common.collect.Maps;
import org.apache.curator.utils.PathUtils;
import org.apache.curator.utils.ThreadUtils;
import org.apache.curator.utils.ZKPaths;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TreeCache
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(TreeCache.class);
    private final boolean createParentNodes;
    private final TreeCacheSelector selector;
    private static final AtomicReferenceFieldUpdater<TreeNode, NodeState> nodeStateUpdater = AtomicReferenceFieldUpdater.newUpdater(TreeNode.class, NodeState.class, "nodeState");
    private static final AtomicReferenceFieldUpdater<TreeNode, ChildData> childDataUpdater = AtomicReferenceFieldUpdater.newUpdater(TreeNode.class, ChildData.class, "childData");
    private static final AtomicReferenceFieldUpdater<TreeNode, ConcurrentMap> childrenUpdater = AtomicReferenceFieldUpdater.newUpdater(TreeNode.class, ConcurrentMap.class, "children");
    private final AtomicLong outstandingOps = new AtomicLong(0L);
    private final AtomicBoolean isInitialized = new AtomicBoolean(false);
    private final TreeNode root;
    private final WatcherRemoveCuratorFramework client;
    private final ExecutorService executorService;
    private final boolean cacheData;
    private final boolean dataIsCompressed;
    private final int maxDepth;
    private final ListenerContainer<TreeCacheListener> listeners = new ListenerContainer();
    private final ListenerContainer<UnhandledErrorListener> errorListeners = new ListenerContainer();
    private final AtomicReference<TreeState> treeState = new AtomicReference<TreeState>(TreeState.LATENT);
    private final ConnectionStateListener connectionStateListener = new ConnectionStateListener(){

        public void stateChanged(CuratorFramework client, ConnectionState newState) {
            TreeCache.this.handleStateChange(newState);
        }
    };
    static final ThreadFactory defaultThreadFactory = ThreadUtils.newThreadFactory((String)"TreeCache");

    public static Builder newBuilder(CuratorFramework client, String path) {
        return new Builder(client, path);
    }

    public TreeCache(CuratorFramework client, String path) {
        this(client, path, true, false, Integer.MAX_VALUE, Executors.newSingleThreadExecutor(defaultThreadFactory), false, new DefaultTreeCacheSelector());
    }

    TreeCache(CuratorFramework client, String path, boolean cacheData, boolean dataIsCompressed, int maxDepth, ExecutorService executorService, boolean createParentNodes, TreeCacheSelector selector) {
        this.createParentNodes = createParentNodes;
        this.selector = (TreeCacheSelector)Preconditions.checkNotNull((Object)selector, (Object)"selector cannot be null");
        this.root = new TreeNode(PathUtils.validatePath((String)path), null);
        Preconditions.checkNotNull((Object)client, (Object)"client cannot be null");
        this.client = client.newWatcherRemoveCuratorFramework();
        this.cacheData = cacheData;
        this.dataIsCompressed = dataIsCompressed;
        this.maxDepth = maxDepth;
        this.executorService = (ExecutorService)Preconditions.checkNotNull((Object)executorService, (Object)"executorService cannot be null");
    }

    public TreeCache start() throws Exception {
        Preconditions.checkState((boolean)this.treeState.compareAndSet(TreeState.LATENT, TreeState.STARTED), (Object)"already started");
        if (this.createParentNodes) {
            this.client.createContainers(this.root.path);
        }
        this.client.getConnectionStateListenable().addListener((Object)this.connectionStateListener);
        if (this.client.getZookeeperClient().isConnected()) {
            this.root.wasCreated();
        }
        return this;
    }

    @Override
    public void close() {
        if (this.treeState.compareAndSet(TreeState.STARTED, TreeState.CLOSED)) {
            this.client.removeWatchers();
            this.client.getConnectionStateListenable().removeListener((Object)this.connectionStateListener);
            this.listeners.clear();
            this.executorService.shutdown();
            try {
                this.root.wasDeleted();
            }
            catch (Exception e) {
                ThreadUtils.checkInterrupted((Throwable)e);
                this.handleException(e);
            }
        }
    }

    public Listenable<TreeCacheListener> getListenable() {
        return this.listeners;
    }

    @VisibleForTesting
    public Listenable<UnhandledErrorListener> getUnhandledErrorListenable() {
        return this.errorListeners;
    }

    private TreeNode find(String findPath) {
        String nextFind;
        PathUtils.validatePath((String)findPath);
        LinkedList rootElements = new LinkedList(ZKPaths.split((String)this.root.path));
        LinkedList findElements = new LinkedList(ZKPaths.split((String)findPath));
        while (!rootElements.isEmpty()) {
            if (findElements.isEmpty()) {
                return null;
            }
            String nextRoot = (String)rootElements.removeFirst();
            nextFind = (String)findElements.removeFirst();
            if (nextFind.equals(nextRoot)) continue;
            return null;
        }
        TreeNode current = this.root;
        while (!findElements.isEmpty()) {
            nextFind = (String)findElements.removeFirst();
            ConcurrentMap<String, TreeNode> map = current.children;
            if (map == null) {
                return null;
            }
            current = (TreeNode)map.get(nextFind);
            if (current != null) continue;
            return null;
        }
        return current;
    }

    public Map<String, ChildData> getCurrentChildren(String fullPath) {
        ImmutableMap result;
        TreeNode node = this.find(fullPath);
        if (node == null || node.nodeState != NodeState.LIVE) {
            return null;
        }
        ConcurrentMap<String, TreeNode> map = node.children;
        if (map == null) {
            result = ImmutableMap.of();
        } else {
            ImmutableMap.Builder builder = ImmutableMap.builder();
            for (Map.Entry entry : map.entrySet()) {
                TreeNode childNode = (TreeNode)entry.getValue();
                ChildData childData = childNode.childData;
                if (childData == null || childNode.nodeState != NodeState.LIVE) continue;
                builder.put(entry.getKey(), (Object)childData);
            }
            result = builder.build();
        }
        return node.nodeState == NodeState.LIVE ? result : null;
    }

    public ChildData getCurrentData(String fullPath) {
        TreeNode node = this.find(fullPath);
        if (node == null || node.nodeState != NodeState.LIVE) {
            return null;
        }
        ChildData result = node.childData;
        return node.nodeState == NodeState.LIVE ? result : null;
    }

    private void callListeners(final TreeCacheEvent event) {
        this.listeners.forEach((Function)new Function<TreeCacheListener, Void>(){

            public Void apply(TreeCacheListener listener) {
                try {
                    listener.childEvent((CuratorFramework)TreeCache.this.client, event);
                }
                catch (Exception e) {
                    ThreadUtils.checkInterrupted((Throwable)e);
                    TreeCache.this.handleException(e);
                }
                return null;
            }
        });
    }

    private void handleException(final Throwable e) {
        if (this.errorListeners.size() == 0) {
            LOG.error("", e);
        } else {
            this.errorListeners.forEach((Function)new Function<UnhandledErrorListener, Void>(){

                public Void apply(UnhandledErrorListener listener) {
                    try {
                        listener.unhandledError("", e);
                    }
                    catch (Exception e2) {
                        ThreadUtils.checkInterrupted((Throwable)e2);
                        LOG.error("Exception handling exception", (Throwable)e2);
                    }
                    return null;
                }
            });
        }
    }

    private void handleStateChange(ConnectionState newState) {
        switch (newState) {
            case SUSPENDED: {
                this.publishEvent(TreeCacheEvent.Type.CONNECTION_SUSPENDED);
                break;
            }
            case LOST: {
                this.isInitialized.set(false);
                this.publishEvent(TreeCacheEvent.Type.CONNECTION_LOST);
                break;
            }
            case CONNECTED: {
                try {
                    this.root.wasCreated();
                }
                catch (Exception e) {
                    ThreadUtils.checkInterrupted((Throwable)e);
                    this.handleException(e);
                }
                break;
            }
            case RECONNECTED: {
                try {
                    this.root.wasReconnected();
                    this.publishEvent(TreeCacheEvent.Type.CONNECTION_RECONNECTED);
                    break;
                }
                catch (Exception e) {
                    ThreadUtils.checkInterrupted((Throwable)e);
                    this.handleException(e);
                }
            }
        }
    }

    private void publishEvent(TreeCacheEvent.Type type) {
        this.publishEvent(new TreeCacheEvent(type, null));
    }

    private void publishEvent(TreeCacheEvent.Type type, String path) {
        this.publishEvent(new TreeCacheEvent(type, new ChildData(path, null, null)));
    }

    private void publishEvent(TreeCacheEvent.Type type, ChildData data) {
        this.publishEvent(new TreeCacheEvent(type, data));
    }

    private void publishEvent(final TreeCacheEvent event) {
        if (this.treeState.get() != TreeState.CLOSED) {
            LOG.debug("publishEvent: {}", (Object)event);
            this.executorService.submit(new Runnable(){

                @Override
                public void run() {
                    try {
                        TreeCache.this.callListeners(event);
                    }
                    catch (Exception e) {
                        ThreadUtils.checkInterrupted((Throwable)e);
                        TreeCache.this.handleException(e);
                    }
                }
            });
        }
    }

    private static enum TreeState {
        LATENT,
        STARTED,
        CLOSED;

    }

    private final class TreeNode
    implements Watcher,
    BackgroundCallback {
        volatile NodeState nodeState = NodeState.PENDING;
        volatile ChildData childData;
        final TreeNode parent;
        final String path;
        volatile ConcurrentMap<String, TreeNode> children;
        final int depth;

        TreeNode(String path, TreeNode parent) {
            this.path = path;
            this.parent = parent;
            this.depth = parent == null ? 0 : parent.depth + 1;
        }

        private void refresh() throws Exception {
            if (this.depth < TreeCache.this.maxDepth && TreeCache.this.selector.traverseChildren(this.path)) {
                TreeCache.this.outstandingOps.addAndGet(2L);
                this.doRefreshData();
                this.doRefreshChildren();
            } else {
                this.refreshData();
            }
        }

        private void refreshChildren() throws Exception {
            if (this.depth < TreeCache.this.maxDepth && TreeCache.this.selector.traverseChildren(this.path)) {
                TreeCache.this.outstandingOps.incrementAndGet();
                this.doRefreshChildren();
            }
        }

        private void refreshData() throws Exception {
            TreeCache.this.outstandingOps.incrementAndGet();
            this.doRefreshData();
        }

        private void doRefreshChildren() throws Exception {
            if (TreeCache.this.treeState.get() == TreeState.STARTED) {
                ((ErrorListenerPathable)((BackgroundPathable)TreeCache.this.client.getChildren().usingWatcher((Watcher)this)).inBackground((BackgroundCallback)this)).forPath(this.path);
            }
        }

        private void doRefreshData() throws Exception {
            if (TreeCache.this.treeState.get() == TreeState.STARTED) {
                if (TreeCache.this.dataIsCompressed) {
                    ((ErrorListenerPathable)((BackgroundPathable)((GetDataWatchBackgroundStatable)TreeCache.this.client.getData().decompressed()).usingWatcher((Watcher)this)).inBackground((BackgroundCallback)this)).forPath(this.path);
                } else {
                    ((ErrorListenerPathable)((BackgroundPathable)TreeCache.this.client.getData().usingWatcher((Watcher)this)).inBackground((BackgroundCallback)this)).forPath(this.path);
                }
            }
        }

        void wasReconnected() throws Exception {
            this.refresh();
            ConcurrentMap<String, TreeNode> childMap = this.children;
            if (childMap != null) {
                for (TreeNode child : childMap.values()) {
                    child.wasReconnected();
                }
            }
        }

        void wasCreated() throws Exception {
            this.refresh();
        }

        void wasDeleted() throws Exception {
            ChildData oldChildData = childDataUpdater.getAndSet(this, null);
            ConcurrentMap childMap = childrenUpdater.getAndSet(this, null);
            if (childMap != null) {
                ArrayList childCopy = new ArrayList(childMap.values());
                childMap.clear();
                for (TreeNode child : childCopy) {
                    child.wasDeleted();
                }
            }
            if (TreeCache.this.treeState.get() == TreeState.CLOSED) {
                return;
            }
            NodeState oldState = nodeStateUpdater.getAndSet(this, NodeState.DEAD);
            if (oldState == NodeState.LIVE) {
                TreeCache.this.publishEvent(TreeCacheEvent.Type.NODE_REMOVED, oldChildData);
            }
            if (this.parent == null) {
                ((ErrorListenerPathable)((BackgroundPathable)TreeCache.this.client.checkExists().usingWatcher((Watcher)this)).inBackground((BackgroundCallback)this)).forPath(this.path);
            } else {
                ConcurrentMap<String, TreeNode> parentChildMap = this.parent.children;
                if (parentChildMap != null) {
                    parentChildMap.remove(ZKPaths.getNodeFromPath((String)this.path), this);
                }
            }
        }

        public void process(WatchedEvent event) {
            LOG.debug("process: {}", (Object)event);
            try {
                switch (event.getType()) {
                    case NodeCreated: {
                        Preconditions.checkState((this.parent == null ? 1 : 0) != 0, (Object)"unexpected NodeCreated on non-root node");
                        this.wasCreated();
                        break;
                    }
                    case NodeChildrenChanged: {
                        this.refreshChildren();
                        break;
                    }
                    case NodeDataChanged: {
                        this.refreshData();
                        break;
                    }
                    case NodeDeleted: {
                        this.wasDeleted();
                    }
                }
            }
            catch (Exception e) {
                ThreadUtils.checkInterrupted((Throwable)e);
                TreeCache.this.handleException(e);
            }
        }

        public void processResult(CuratorFramework client, CuratorEvent event) throws Exception {
            LOG.debug("processResult: {}", (Object)event);
            Stat newStat = event.getStat();
            switch (event.getType()) {
                case EXISTS: {
                    Preconditions.checkState((this.parent == null ? 1 : 0) != 0, (Object)"unexpected EXISTS on non-root node");
                    if (event.getResultCode() != KeeperException.Code.OK.intValue()) break;
                    nodeStateUpdater.compareAndSet(this, NodeState.DEAD, NodeState.PENDING);
                    this.wasCreated();
                    break;
                }
                case CHILDREN: {
                    if (event.getResultCode() == KeeperException.Code.OK.intValue()) {
                        ChildData oldChildData = this.childData;
                        if (oldChildData != null && oldChildData.getStat().getMzxid() == newStat.getMzxid()) {
                            childDataUpdater.compareAndSet(this, oldChildData, new ChildData(oldChildData.getPath(), newStat, oldChildData.getData()));
                        }
                        if (event.getChildren().isEmpty()) break;
                        ConcurrentMap<String, TreeNode> childMap = this.children;
                        if (childMap == null) {
                            childMap = Maps.newConcurrentMap();
                            if (!childrenUpdater.compareAndSet(this, null, childMap)) {
                                childMap = this.children;
                            }
                        }
                        ArrayList<String> newChildren = new ArrayList<String>();
                        for (String child : event.getChildren()) {
                            if (childMap.containsKey(child) || !TreeCache.this.selector.acceptChild(ZKPaths.makePath((String)this.path, (String)child))) continue;
                            newChildren.add(child);
                        }
                        Collections.sort(newChildren);
                        for (String child : newChildren) {
                            String fullPath;
                            TreeNode node;
                            if (childMap.putIfAbsent(child, node = new TreeNode(fullPath = ZKPaths.makePath((String)this.path, (String)child), this)) != null) continue;
                            node.wasCreated();
                        }
                        break;
                    }
                    if (event.getResultCode() != KeeperException.Code.NONODE.intValue()) break;
                    this.wasDeleted();
                    break;
                }
                case GET_DATA: {
                    if (event.getResultCode() == KeeperException.Code.OK.intValue()) {
                        boolean added;
                        ChildData toPublish = new ChildData(event.getPath(), newStat, event.getData());
                        ChildData oldChildData = TreeCache.this.cacheData ? childDataUpdater.getAndSet(this, toPublish) : childDataUpdater.getAndSet(this, new ChildData(event.getPath(), newStat, null));
                        if (this.parent == null) {
                            added = nodeStateUpdater.getAndSet(this, NodeState.LIVE) != NodeState.LIVE;
                        } else {
                            added = nodeStateUpdater.compareAndSet(this, NodeState.PENDING, NodeState.LIVE);
                            if (!added && this.nodeState != NodeState.LIVE) {
                                return;
                            }
                        }
                        if (added) {
                            TreeCache.this.publishEvent(TreeCacheEvent.Type.NODE_ADDED, toPublish);
                            break;
                        }
                        if (oldChildData != null && oldChildData.getStat().getMzxid() == newStat.getMzxid()) break;
                        TreeCache.this.publishEvent(TreeCacheEvent.Type.NODE_UPDATED, toPublish);
                        break;
                    }
                    if (event.getResultCode() != KeeperException.Code.NONODE.intValue()) break;
                    this.wasDeleted();
                    break;
                }
                default: {
                    LOG.info(String.format("Unknown event %s", event));
                    TreeCache.this.outstandingOps.decrementAndGet();
                    return;
                }
            }
            if (TreeCache.this.outstandingOps.decrementAndGet() == 0L && TreeCache.this.isInitialized.compareAndSet(false, true)) {
                TreeCache.this.publishEvent(TreeCacheEvent.Type.INITIALIZED);
            }
        }
    }

    private static enum NodeState {
        PENDING,
        LIVE,
        DEAD;

    }

    public static final class Builder {
        private final CuratorFramework client;
        private final String path;
        private boolean cacheData = true;
        private boolean dataIsCompressed = false;
        private ExecutorService executorService = null;
        private int maxDepth = Integer.MAX_VALUE;
        private boolean createParentNodes = false;
        private TreeCacheSelector selector = new DefaultTreeCacheSelector();

        private Builder(CuratorFramework client, String path) {
            this.client = (CuratorFramework)Preconditions.checkNotNull((Object)client);
            this.path = PathUtils.validatePath((String)path);
        }

        public TreeCache build() {
            ExecutorService executor = this.executorService;
            if (executor == null) {
                executor = Executors.newSingleThreadExecutor(defaultThreadFactory);
            }
            return new TreeCache(this.client, this.path, this.cacheData, this.dataIsCompressed, this.maxDepth, executor, this.createParentNodes, this.selector);
        }

        public Builder setCacheData(boolean cacheData) {
            this.cacheData = cacheData;
            return this;
        }

        public Builder setDataIsCompressed(boolean dataIsCompressed) {
            this.dataIsCompressed = dataIsCompressed;
            return this;
        }

        public Builder setExecutor(ThreadFactory threadFactory) {
            return this.setExecutor(Executors.newSingleThreadExecutor(threadFactory));
        }

        public Builder setExecutor(ExecutorService executorService) {
            this.executorService = (ExecutorService)Preconditions.checkNotNull((Object)executorService);
            return this;
        }

        public Builder setMaxDepth(int maxDepth) {
            this.maxDepth = maxDepth;
            return this;
        }

        public Builder setCreateParentNodes(boolean createParentNodes) {
            this.createParentNodes = createParentNodes;
            return this;
        }

        public Builder setSelector(TreeCacheSelector selector) {
            this.selector = selector;
            return this;
        }
    }
}

